<Query Kind="Program">
  <Reference>&lt;RuntimeDirectory&gt;\System.Net.Http.dll</Reference>
  <NuGetReference>Newtonsoft.Json</NuGetReference>
  <Namespace>System.Net.Http</Namespace>
  <Namespace>System.Net.Http.Headers</Namespace>
  <Namespace>Newtonsoft.Json</Namespace>
  <Namespace>System.Net</Namespace>
  <Namespace>System.Threading.Tasks</Namespace>
</Query>

// Example of how to use bilinfo CarApi for QA01 tested with LINQPAD 7 Dotnet Core 6.0.8
// Remember to update the version in Car.json if you want to make updates to the car

// Prerequisites:
// 1. Username and Password for the dealer in Bilinfo net you need to manage cars for. Fill out in the variable BilinfoNetEmail and BilinfoNetPassword
// 2. ClientId and ClientSecret for Bilinfo IdentityServer. Fill out in the variable IdentityServerClientId and IdentityServerClientSecret
// 3. Azure subscriptionkey with access to carapi. Fill out in the variable SubscriptionKey
// 4. DealerId (Guid) for the dealer you want to create a car for. Fill out in the variable DealerId. The Dealer must have a Bilinfo Pro subscription package
// 5. Redirecturl url whitelisted when creating the ClientId. This should be a domain that you own. Fill out in the variable RedirectUrl

const string CarapiUrl = "https://publicapi.bilinfo.net/carapiqa/";
const string BilinfoUrl = "https://www.qa01.bilinfo.net";

const string BilinfoNetEmail = <<INSERT BILINFO USERNAME HERE>>;
string BilinfoNetPassword = Util.GetPassword("BilinfoNetPassword");
const string IdentityServerClientId = <<INSERT IDENTITYSERVER CLIENT ID>>;
string IdentityServerClientSecret = Util.GetPassword("BilinfoIdentityServerClientServer");
const string SubscriptionKey = <<INSERT SUBSCRIPTIONKEY HERE>>;
const string DealerId = <<INSERT DEALERID HERE>>;
const string RedirectUrl = <<INSERT REDIRECTURL HERE>>;
const string IdentityServer = "https://qa01-accounts.ecgh.dk/oauth/connect/token";

HttpClient CarApiHttpClient = CreateHttpClient();

static HttpClient CreateHttpClient()
{
	var httpClient = new HttpClient();
	httpClient.DefaultRequestHeaders.Accept.Clear();
	httpClient.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/json"));
	httpClient.DefaultRequestHeaders.Add("Ocp-Apim-Subscription-Key",SubscriptionKey);

	return httpClient;
}

public async Task Main()
{
	var bearerTokenGenerator = new UserQuery.BilinfoTokenGenerator(BilinfoUrl, BilinfoNetEmail, BilinfoNetPassword, IdentityServerClientId, IdentityServerClientSecret);
	var bearerToken = await bearerTokenGenerator.GetBilinofAuthToken();
	
	var carApiUrl = new Uri(CarapiUrl);
	await CreateOrUpdateCar(carApiUrl.ToString(), bearerToken);
}

async Task CreateOrUpdateCar(string carApiUrl, string bearerToken)
{
	var operationId = await CreateOrUpdateCarRequest(carApiUrl + $"api/dealers/{DealerId}/cars/", bearerToken);
	operationId.Dump();	
}

async Task<OperationIdResponse> CreateOrUpdateCarRequest(string url, string bearerToken)
{
	using (var httpRequestMessage = new HttpRequestMessage(HttpMethod.Post,	new Uri(url)))
	{
		AddToken(httpRequestMessage, bearerToken);
		AddCarJson(httpRequestMessage);

		using (var httpResponseMessage = await CarApiHttpClient.SendAsync(httpRequestMessage))
		{
			var json = await httpResponseMessage.Content.ReadAsStringAsync();
			return JsonConvert.DeserializeObject<OperationIdResponse>(json);
		}
	}
}

void AddCarJson(HttpRequestMessage httpRequestMessage)
{
	string carJsonPath = Path.GetDirectoryName(Util.CurrentQueryPath) + @"\car.json";
	var car = File.ReadAllText(carJsonPath);
	httpRequestMessage.Content = new StringContent(car, Encoding.UTF8, "application/json");
}

void AddToken(HttpRequestMessage httpRequestMessage, string bearerToken)
{
	httpRequestMessage.Headers.Authorization = new AuthenticationHeaderValue("Bearer", bearerToken);
}

async Task<AccessToken> RequestToken(string clientId, string clientSecret, string scope)
{
	var basicAuth = Convert.ToBase64String(System.Text.Encoding.ASCII.GetBytes(clientId + ':' + clientSecret));

	using (var httpRequestMessage = new HttpRequestMessage(HttpMethod.Post, IdentityServer))
	{
		httpRequestMessage.Headers.Authorization = new AuthenticationHeaderValue("Basic", basicAuth);

		httpRequestMessage.Content = new FormUrlEncodedContent(new Dictionary<string, string>
			{
				{ "grant_type", "client_credentials" },
				{ "scope", scope },
			});

		using (var httpResponseMessage = await CarApiHttpClient.SendAsync(httpRequestMessage))
		{
			var json = await httpResponseMessage.Content.ReadAsStringAsync();
			return JsonConvert.DeserializeObject<AccessToken>(json);
		}
	}
}

class OperationIdResponse
{
	[JsonProperty("operationId")]
	public string OperationId { get; set; }
}
class OperationStatusResponse
{
	[JsonProperty("status")]
	public int Status { get; set; }

	[JsonProperty("statusMessage")]
	public string StatusMessage { get; set; }
}
class AccessToken
{
	[JsonProperty("access_token")]
	public string Token { get; set; }

	[JsonProperty("token_type")]
	public string Type { get; set; }

	[JsonProperty("expires_in")]
	public int ExpiresIn { get; set; }
}


public class BilinfoTokenGenerator
{
	private string _bilinfoUrl;
	private string _userName;
	private string _password;
	private string _identityServerClientId;
	private string _identityServerClientSecret;

	public BilinfoTokenGenerator(string bilinfoUrl, string userName, string password, string identityServerClientId, string identityServerClientSecret)
	{
		_bilinfoUrl = bilinfoUrl;
		_userName = userName;
		_password = password;
		_identityServerClientId = identityServerClientId;
		_identityServerClientSecret = identityServerClientSecret;
	}
	public async Task<string> GetBilinofAuthToken()
	{
		CookieContainer cookies = new CookieContainer();
		HttpClientHandler handler = new HttpClientHandler();
		handler.AllowAutoRedirect = false;
		handler.CookieContainer = cookies;

		var postData = new List<KeyValuePair<string, string>>();
		postData.Add(new KeyValuePair<string, string>("username", _userName));
		postData.Add(new KeyValuePair<string, string>("Password", _password));

		HttpContent content = new FormUrlEncodedContent(postData);

		var client = new HttpClient(handler);
		client.DefaultRequestHeaders.Clear();
		client.DefaultRequestHeaders.TryAddWithoutValidation("User-Agent", "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.17 (KHTML, like Gecko) Chrome/24.0.1312.57 Safari/537.17");
		client.PostAsync(_bilinfoUrl + "/Account/Login", content).GetAwaiter().GetResult();

		return await GetAccessToken(client);
	}
	private async Task<string> GetAccessToken(HttpClient client)
	{
		var getClientCodeUrl = _bilinfoUrl + $"/oauth/connect/authorize?client_id={IdentityServerClientId}&redirect_uri={RedirectUrl}&scope=openid%20dealer%20https://carapi.ecgh.dk/auth/write%20https://carapi.ecgh.dk/auth/read&response_type=code&nonce=" + Guid.NewGuid().ToString();

		var signinUrl = client.GetAsync(getClientCodeUrl).GetAwaiter().GetResult();

		string urlWithCode;

		if (signinUrl.Headers.Location.AbsoluteUri.Contains("signin"))
		{
			//call signin
			var autorizeAgain = client.GetAsync(signinUrl.Headers.Location.AbsoluteUri).GetAwaiter().GetResult();

			var responseWithCode = await client.GetAsync(autorizeAgain.Headers.Location.AbsoluteUri);
			urlWithCode = responseWithCode.Headers.Location.AbsoluteUri;
		}
		else
		{
			urlWithCode = signinUrl.Headers.Location.AbsoluteUri;
		}
		//get the code from the returned url
		var code = Regex.Match(urlWithCode, "(?<=code=)(.*)(?=&)").Value;

		return await GetToken(client, code);
	}
	private async Task<string> GetToken(HttpClient client, string code)
	{
		var tokenPostData = new List<KeyValuePair<string, string>>();
		tokenPostData.Add(new KeyValuePair<string, string>("grant_type", "authorization_code"));
		tokenPostData.Add(new KeyValuePair<string, string>("code", code));
		tokenPostData.Add(new KeyValuePair<string, string>("redirect_uri", RedirectUrl));
		tokenPostData.Add(new KeyValuePair<string, string>("scope", "dealer"));

		client.DefaultRequestHeaders.Clear();
		client.DefaultRequestHeaders.TryAddWithoutValidation("Authorization", $"Basic {Base64Encode(_identityServerClientId + ":" + _identityServerClientSecret )}");
		client.DefaultRequestHeaders.TryAddWithoutValidation("Accept", "application/json");

		var responseWithToken = await client.PostAsync(_bilinfoUrl + "/oauth/connect/token", new FormUrlEncodedContent(tokenPostData));
		var resultWithToken = await responseWithToken.Content.ReadAsStringAsync();

		return Regex.Match(resultWithToken, "(?<=access_token\".*:.*\")(.*)(?=\",)").Value;
	}

	private string Base64Encode(string plainText)
	{
		var plainTextBytes = System.Text.Encoding.UTF8.GetBytes(plainText);
		return System.Convert.ToBase64String(plainTextBytes);
	}
}